
/* polar coordinate system interpolation, azimuthal and hankel transforms                 */ 
/* written by Douglas C. Braun                                                            */
/* Use and modify freely but please inform me at dbraun@cora.nwra.com of questions        */
/* or major improvements and acknowlege the use of this code and its author in any        */
/* relevant publications                                                                  */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <math.h>
#include "hank_hdr.h"
#include "fitssubs.h"
#include "fitssubs.c"
#include "params.h"

#define   PI             3.1415926
#define   DEGPERRAD     57.29577951

main(int argc, char *argv[])
{
    FILE *fs;
    short *pin, **ppin;
    short *pin_frame, **ppin_frame;
    float *pfin, **ppfin;
    float *px, **ppx, *py, **ppy;
    float *aztr, **paztr;
    float *htr_re, **phtr_re, *htr_im, **phtr_im;
    float dx, dy, dx2, dy2, value, phi, dphi;
    float sinphi[MMAX_H][NPHI], cosphi[MMAX_H][NPHI], norm;
    float x, y, x0, y0;
    float iix, iix2, iiy, iiy2, r2, rs2;
    float *phi_dat;
    float rthet, theta, xmin;
    float c[1000];
    float win, wtheta[1000], num, fact;
    float lt, ll[1000];
    float jrterm, yrterm, jiterm, yiterm;
    int i, nbr, nbw, fdi, fdo, numh, nim, nim1, nim2, bitpixh, imsize, imsizeh, numpix;
    int row, col, acol, it, ip, ix, iy, l, m, naz, nthet;
    int naxis1h, naxis2h, naxis3h;
    int mode = 0644;
    long bytpix, lastbs, n, nbpad;
    long offset, offset_re, offset_im;
    char *zpad;
    char *header, *header_out;
    char *comment;
    void make_header(char *header, int bitpixh, int naxis1h, int naxis2h, int naxis3h, float x0, float y0, char *comment);
    float ***jbess, ***ybess;
    float **array2d(int nrows, int ncols);

    comment = (char *) malloc(20);
    header = (char *) malloc(BSIZE);
    header_out = (char *) malloc(BSIZE);

    if (argc != 5 && argc != 7)  {
        printf("Usage: %s INPUT OUTPUT X0 Y0 [nframe_start nframe_end]\n", argv[0] );
        exit(1);
    } 
    sscanf(argv[3], "%f", &x0); 
    sscanf(argv[4], "%f", &y0); 
    if (argc==7) sscanf(argv[5], "%d", &nim1);
    if (argc==7) sscanf(argv[6], "%d", &nim2);

/* read headers and grab first one */
    numh = readhdr(argv[1]);
    readhdr1(argv[1], header);
    if (argc==5) nim1=0;
    if (argc==5) nim2=naxis3-1;
    naxis3h=nim2+1-nim1;
    if (bitpix != 16 && bitpix != -32) {
        fprintf(stderr, "bitpix = %5d\n", bitpix);
        exit(1);
    }
    if (naxis != 3 ) {
        fprintf(stderr, "naxis = %5d -- should be 3\n", naxis);
        exit(1);
    }
    if (bscale == 0) {
        bscale = 1.;
        b_zero = 0.;
    }
    imsize=naxis1*naxis2*abs(bitpix/8);

    nthet = NTHET_H2-NTHET_H1;
    naz = 2*MMAX_H + 1;
    tsam = TSAM;

    naxis1h = NLMAX+1;
    naxis2h = naz;
    imsizeh=naxis1h*naxis2h*sizeof(float);

    aztr = (float *) malloc(naz*nthet*sizeof(float));
    paztr = (float **) malloc(nthet*sizeof(float *));
    for (row = 0; row < nthet; row++) paztr[row] = aztr + (row*naz);

    numpix = naxis1*naxis2;
    pin = (short *) malloc(numpix*sizeof(short));
    ppin = (short **) malloc(naxis2*sizeof(short *));
    for (row = 0; row < naxis2; row++) ppin[row] = pin + (row*naxis1);

    pfin = (float *) malloc(numpix*sizeof(float));
    ppfin = (float **) malloc(naxis2*sizeof(float *));
    for (row = 0; row < naxis2; row++) ppfin[row] = pfin + (row*naxis1);

    pin_frame = (short *) malloc(NPHI*nthet*sizeof(short));
    ppin_frame = (short **) malloc(nthet*sizeof(short *));
    for (row = 0; row < nthet; row++) ppin_frame[row] = pin_frame + (row*NPHI);

    px = (float *) malloc(NPHI*nthet*sizeof(float));
    ppx = (float **) malloc(nthet*sizeof(float *));
    for (row = 0; row < nthet; row++) ppx[row] = px + (row*NPHI);
 
    py = (float *) malloc(NPHI*nthet*sizeof(float));
    ppy = (float **) malloc(nthet*sizeof(float *));
    for (row = 0; row < nthet; row++) ppy[row] = py + (row*NPHI);
 
    phi_dat = (float *)malloc(NPHI*sizeof(float));
    dphi = 2.*PI/(float)(NPHI);

    htr_re = (float *) malloc(naxis1h*naxis2h*sizeof(float));
    phtr_re = (float **) malloc(naxis2h*sizeof(float *));
    for (row=0; row < naxis2h; row++)  phtr_re[row] = htr_re + (row*naxis1h);
    htr_im = (float *) malloc(naxis1h*naxis2h*sizeof(float));
    phtr_im = (float **) malloc(naxis2h*sizeof(float *));
    for (row=0; row < naxis2h; row++)  phtr_im[row] = htr_im + (row*naxis1h);
    
    jbess = (float ***) malloc((unsigned) (MMAX_H+1)*sizeof(float **));
    ybess = (float ***) malloc((unsigned) (MMAX_H+1)*sizeof(float **));
    for (m=0; m <= MMAX_H; m++) {
        jbess[m] = array2d(NTHET_H2+1, naxis1h);
        ybess[m] = array2d(NTHET_H2+1, naxis1h);
    }

    norm=(float)NPHI;
    for (m=1; m <= MMAX_H; m++ ) {
        phi = 0.;
        for (ip = 0; ip < NPHI; ip ++) {
            cosphi[m-1][ip] = cos((float)(m)*phi)/norm;
            sinphi[m-1][ip] = sin((float)(m)*phi)/norm;
            phi += dphi;
        }
    }

    for (it = 0; it < nthet; it++) {
        for (ip = 0; ip < NPHI; ip ++) {
            ppin_frame[it][ip] = 1;
            phi = dphi*(float)(ip);
            ppx[it][ip] = x0 + (float)(it+NTHET_H1)*cos(phi); 
            ppy[it][ip] = y0 + (float)(it+NTHET_H1)*sin(phi); 
            ix = (int) ppx[it][ip];
            iy = (int) ppy[it][ip];
            if (ix < 0 || ix > naxis1-2) ppin_frame[it][ip] = 0;
            if (iy < 0 || iy > naxis2-2) ppin_frame[it][ip] = 0;
        }
    }

    numout=0;
    for (it = 0; it < nthet; it++) {
        for (ip = 0; ip < NPHI; ip ++) {
            if (ppin_frame[it][ip] == 0) numout++;
        }
    }
    rthet = (float)(NTHET_H2-NTHET_H1)*DTHET;
    deltal = 2.*PI/rthet;
    for (l=NLMIN; l <=NLMAX; l ++) {
        ll[l] = (float)(l)*deltal;
        c[l] = PI*ll[l]/(2.*(float)(NTHET_H2-NTHET_H1));
    }

/* calculate window function     */
    num = (float) (NTHET_H2-NTHET_H1);
    for (it=NTHET_H1; it<NTHET_H2; it++) {
        y=(float) (it- NTHET_H1);
        fact = (y-0.5*(num-1.))/(0.5*(num+1.));
        win = 1. - fact*fact;
        theta = (float)(it)*DTHET;
        wtheta[it]=win*theta;
    }

/* calculate Bessel functions */
    for (m=0; m <= MMAX_H; m++) {
        for (l=NLMIN; l <=NLMAX; l ++) {
            xmin = ll[l]*NTHET_H1*DTHET;
            if (xmin < ((float)(m) - 0.5) ) {
                for (it=NTHET_H1; it<NTHET_H2; it++) {
                    jbess[m][it][l] = 0.;
                    ybess[m][it][l] = 0.;
                }
            }
            else {
                for (it=NTHET_H1; it<NTHET_H2; it++) {
                    theta = (float)(it)*DTHET;
                    x = ll[l]*theta;
                    jbess[m][it][l] = jn(m, x)*wtheta[it];
                    ybess[m][it][l] = yn(m, x)*wtheta[it];
                }
            }
        }
    }

    bitpixh=-32;
    daxis1=deltal;
    daxis2=1.;
    daxis3=tsam;
    daxis4=1.;
    fdi = open (argv[1], O_RDONLY);
    fdo = open (argv[2], O_RDWR | O_CREAT, mode);
    make_header(header_out, bitpixh, naxis1h, naxis2h, naxis3h, x0, y0, "TARGET");
    nbw = write(fdo, header_out, BSIZE);

/*   start image loop here */
    for (nim = nim1; nim <= nim2 ; nim ++) { 
        offset = BSIZE*numh + (nim*imsize);
        offset_re = BSIZE + ((nim-nim1)*imsizeh);
        offset_im = BSIZE + ((nim+naxis3h-nim1)*imsizeh);
        lseek(fdi, offset, 0);
        if (bitpix == 16) {
            if ( (nbr = read(fdi, pin, imsize)) != imsize) {
                fprintf(stderr, "Error extracting frame %5d: %d bytes returned\n",nim, nbr);
                exit(1);
            }
            swab(pin, pin, imsize);
            for (i= 0; i < numpix; i++) { 
                pfin[i] = ( b_zero + bscale*(float)pin[i] );
            }
        }
        if (bitpix == -32) {
            if ( (nbr = read(fdi, pfin, imsize)) != imsize) {
                fprintf(stderr, "Error extracting frame %5d: %d bytes returned\n",nim, nbr);
                exit(1);
            }
            iswab(pfin, pfin, imsize);
        }

        for (it = 0; it < nthet; it++) {
            for (m=0; m < naz; m++)  paztr[it][m] = 0.;
            for (ip = 0; ip < NPHI; ip ++) {
                phi_dat[ip] = 0.;
                if (ppin_frame[it][ip]) {
                    x= ppx[it][ip];
                    y= ppy[it][ip];
                    ix = (int) x;
                    iy = (int) y;
                    dx = x - (float) ix;
                    dy = y - (float) iy;
                    dx2 = 1. - dx;
                    dy2 = 1. - dy;
                    value = ppfin[iy][ix]*dx2*dy2 +
                        ppfin[iy+1][ix]*dx2*dy +
                        ppfin[iy][ix+1]*dx*dy2 +
                        ppfin[iy+1][ix+1]*dx*dy;
                    phi_dat[ip] = value;
                }
                else {
                    phi_dat[ip] = 0.;
                }
                paztr[it][0] += phi_dat[ip]/norm;
                for (m=1; m <= MMAX_H; m++) {
                    paztr[it][2*m-1] += phi_dat[ip]*cosphi[m-1][ip];
                    paztr[it][2*m]   += phi_dat[ip]*sinphi[m-1][ip];
                }
            }
        }

        for (l=NLMIN; l <=NLMAX; l++) {
            for (i=0; i < naxis2h; i++) {
                phtr_re[i][l] = 0.;
                phtr_im[i][l] = 0.;
            }
            for (it=NTHET_H1; it<NTHET_H2; it++) {
                theta = (float)(it)*DTHET;
                jrterm=paztr[it-NTHET_H1][0]*jbess[0][it][l];
                yrterm=paztr[it-NTHET_H1][0]*ybess[0][it][l];
                phtr_re[ 0][l] += jrterm;
                phtr_im[ 0][l] -= yrterm;
                for (m=1; m <= MMAX_H; m++) {
                    acol = 1+2*(m-1);
                    jrterm=paztr[it-NTHET_H1][acol]*jbess[m][it][l];
                    yrterm=paztr[it-NTHET_H1][acol]*ybess[m][it][l];
                    jiterm=paztr[it-NTHET_H1][acol+1]*jbess[m][it][l];
                    yiterm=paztr[it-NTHET_H1][acol+1]*ybess[m][it][l];
                    phtr_re[m][l]         +=  jrterm + yiterm;
                    phtr_im[m][l]         +=  jiterm - yrterm;
                    phtr_re[naxis2h-m][l] +=  jrterm - yiterm;
                    phtr_im[naxis2h-m][l] += -jiterm - yrterm;
                }
            } /* end loop in theta */
            for (i=0; i < naxis2h; i++) {
                phtr_re[i][l] *= c[l];
                phtr_im[i][l] *= c[l];
            }
        } /* end loop in L */

        iswab(htr_re,htr_re,imsizeh);
        lseek(fdo, offset_re, 0);
        if ( (nbw = write(fdo, htr_re, imsizeh)) != imsizeh) {
            fprintf(stderr, "Error writing real frame nim = %5d: %d bytes returned\n",nim, nbw);
            exit(1);
        }
        iswab(htr_im,htr_im,imsizeh);
        lseek(fdo, offset_im, 0);
        if ( (nbw = write(fdo, htr_im, imsizeh)) != imsizeh) {
            fprintf(stderr, "Error writing imag frame nim = %5d: %d bytes returned\n",nim, nbw);
            exit(1);
        }
    }
    close(fdi);

    bytpix=abs(bitpix)/8;
    lastbs = (bytpix*naxis1h*naxis2h*naxis3*2)%BSIZE;
    if (lastbs > 0) {
        nbpad = BSIZE-lastbs;
        zpad=(char *)malloc(nbpad);
        offset = BSIZE+ (naxis3*2*imsizeh);
        for (n=0; n < nbpad; n++) zpad[n] = 0;
        lseek(fdo, offset, 0);
        if ( (nbw = write (fdo, zpad, nbpad)) != nbpad) {
          fprintf(stderr, "%s: Error writing zero-pad in %s - returned %d bytes\n", argv[0], argv[2], nbw);
          exit(1);
        }
    }
    close(fdo);

    free(zpad);
    free(ybess);
    free(jbess);
    free(phtr_im);
    free(htr_im);
    free(phtr_re);
    free(htr_re);
    free(phi_dat);
    free(ppy);
    free(py);
    free(ppx);
    free(px);
    free(ppin_frame);
    free(pin_frame);
    free(ppfin);
    free(pfin);
    free(ppin);
    free(pin);
    free(paztr);
    free(aztr);
    free(header_out);
    free(header);
}

void make_header(char *header, int bitpixh, int naxis1h, int naxis2h, int naxis3h, float x0, float y0, char *comment)
{
    int nk, nhb;
    int offset;

/* initialize  header array to asci blanks */
    for (nhb = 0; nhb < BSIZE; nhb++) 
        header[nhb] = ' ';

    offset = 0;
    sprintf(header,          "SIMPLE  =%21c", simple);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "BITPIX  =%21i", bitpixh);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "COMMENT =%21s", comment);
    header[offset + 30] = ' ';

/*
    offset += 80;
    sprintf(header + offset, "BSCALE  =%21e", bscale);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "BZERO   =%21e", b_zero);
    header[offset + 30] = ' ';
*/

    offset += 80;
    sprintf(header + offset, "NAXIS   =%21i", 4);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NAXIS1  =%21i", naxis1h);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NAXIS2  =%21i", naxis2h);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NAXIS3  =%21i", naxis3h);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NAXIS4  =%21i", 2);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "DAXIS1  =%21.4e", daxis1);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "DAXIS2  =%21.4e", daxis2);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "DAXIS3  =%21.4e", daxis3);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "DAXIS4  =%21.4e", daxis4);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "X0      =%21.4e", x0);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "Y0      =%21.4e", y0);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "MAXM    =%21i", MMAX_H);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NTHET1  =%21i", NTHET_H1);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NTHET2  =%21i", NTHET_H2);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "DTHETA  =%21.4e", DTHET);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "DELTAL  =%21.4e", deltal);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NLSTART =%21i", NLMIN);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NLSTOP  =%21i", NLMAX);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NLTOT   =%21i", 1+NLMAX);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "END");
    header[offset + 3] = ' ';

}

float **array2d(int nrows, int ncols)
{
    int i;
    float **a;
    a = (float **) malloc((unsigned)(nrows)*sizeof(float *));
    for (i=0; i < nrows; i++) {
        a[i] = (float *) malloc((unsigned)(ncols)*sizeof(float));
    }
    return a;
}
