#include <stdlib.h>
#include <string.h>
#include <arpa/inet.h>
#include <unistd.h>

/* General FITS routines for processing FITS images */
/* written by Douglas C. Braun                                                            */
/* Use and modify freely but please inform me at dbraun@cora.nwra.com of questions        */
/* or major improvements and acknowlege the use of this code and its author in any        */
/* relevant publications                                                                  */


int rhdr( char *header )
/* reads keyword values into appropriate global variables */
/*  returns 1 if END keyword found, else returns 0        */
{
    int end = 0;
    char line[80];
    char kwd[80];
    int nline, n, offset;
    for( nline=0; nline < 36; nline++) {  
        offset = nline * 80;
        strncpy(line, &header[offset], 80);
        sscanf(line, "%s", kwd); 
        if ( strcmp(kwd, "END") == 0)  {
            end = 1;
            break;
        }
        for (n=0; n < NKEYWDS; n++) {
        if( strncmp(kwd, hdr[n].keyword, 8) == 0)
            sscanf(line, hdr[n].datfmt, hdr[n].dat);  
        }
    }
    return( end );
}

int endline( char *header )
/*  returns line number of END keyword if found, else returns -1 */
{
    int end = -1;
    char line[80];
    char kwd[80];
    int nline,  offset;
    for( nline=0; nline < 36; nline++) {  
        offset = nline * 80;
        strncpy(line, &header[offset], 80);
        sscanf(line, "%s", kwd); 
        if ( strcmp(kwd, "END") == 0)  {
            end = nline+1;
            break;
        }
    }
    return( end );
}

int findkwd( char *wkwd, char *header )
/*  returns line number of wanted keyword if found, else returns -1 */
{
    int found = -1;
    char line[80];
    char kwd[80];
    int nline,  offset;
    for( nline=0; nline < 36; nline++) {  
        offset = nline * 80;
        strncpy(line, &header[offset], 80);
        sscanf(line, "%s", kwd); 
        if ( strcmp(kwd, wkwd) == 0)  {
            found = nline+1;
            break;
        }
    }
    return( found );
}

int roundup( float f)
/* roundups floating point to integer */
{
    int i;
    i = (int) f;
    if ( (float) i == f ) return (i);
    else return (i+1);
}

/*                                                            */
/*   Main subroutines for reading/writing FITS images on disk */

int readhdr (char *fname) 

/* reads fits header from disk to get image size */
/* returns number of header records found        */

{
    int end = 0;
    int nbyter, fdi;
    int numh=0;
     
    char *header;

    simple = '\0';
    header = (char *) malloc (BSIZE);
    fdi = open (fname, O_RDONLY );
    if (fdi < 0) {
        fprintf(stderr, "readhdr: Error opening file %s\n", fname);
        exit(1);
    }
    while (!end) {
        nbyter = read (fdi, header, BSIZE);
        if (nbyter != BSIZE) {
            fprintf(stderr, "readhdr: Error reading header %5d\n", nbyter);
            exit(1);
        }
        numh++;
        if (numh > 100) {
            fprintf(stderr, "readhdr: Exceeded 100 header limit\n");
            exit(1);
        }
        end = rhdr( header ); 
    }
    close (fdi); /* Close disk file. */
    free (header);
    if (simple != 'T') {
        fprintf(stderr, "readhdr: simple = %c in %s\n", simple, fname);
        exit(1);
    }
    if (naxis==1) {
        naxis=2;
        naxis2=1;
    }
    return(numh);
}

void readhdr1 (char *fname, char *header)
/* reads first header into character array */
{
    int fdi;
    int nbyter; 

    fdi = open (fname, O_RDONLY );
    if (fdi < 0) {
        fprintf(stderr, "readhdr1: Error opening file %s\n", fname);
        exit(1);
    }

    nbyter = read (fdi, header, BSIZE);
    if (nbyter != BSIZE) {
        fprintf(stderr, "readhdr1: Error reading first header in %s: %d\n", 
                 fname, nbyter);
        exit(1);
    }
    close(fdi);
}

void readimg (char *fname, void *imarr)
/* reads fits disk file into character array */
{
    int end = 0;
    int fdi;
    int nbyter, imsize, numh=0; 
    char *header;

    header = (char *) malloc (BSIZE);
    fdi = open (fname, O_RDONLY );
    if (fdi < 0) {
        fprintf(stderr, "readimg: Error opening file %s\n", fname);
        exit(1);
    }

    /* skip headers */
    while (!end) {
        nbyter = read (fdi, header, BSIZE);
        if (nbyter != BSIZE) {
            fprintf(stderr, "readimg: Error reading header %5d in %s\n", 
                     nbyter, fname);
            exit(1);
        }
        numh++;
        if (numh > 100) {
            fprintf(stderr, "readimg: Exceeded 100 record limit in %s\n",
                    fname);
            exit(1);
        }
        end = rhdr( header);
    }
    if (naxis==1) {
        naxis=2;
        naxis2=1;
    }
    if ( bitpix != 16 && bitpix != 32 && bitpix != -32) {
        fprintf(stderr, "readimg: bitpix = %5d not implemented\n", bitpix );
        exit(1);
    }
    imsize = abs(bitpix)*naxis1*naxis2/8;
    if (naxis > 2) imsize *= naxis3;
    if (naxis > 3) imsize *= naxis4;
    if ( (nbyter = read (fdi, imarr, imsize)) != imsize) {
        fprintf(stderr, "readimg: Error reading image %s - returned %d bytes\n", fname, nbyter);
        exit(1);
    }
    if (bitpix == 16 && 12345 != htons(12345)) swab(imarr, imarr, imsize);
    if (abs(bitpix) == 32 && 123456789 != htonl(123456789)) iswab(imarr, imarr, imsize); 
    close (fdi); /* Close disk file. */
    free (header);
}

void writeimg (char *fname, char *header, void *imarr)
/* writes data array into fits disk file                                */
/* 1/13/95: pads data with zeros to have complete 2880 byte records     */
/*          Files written prior to this DO NOT CONFORM TO THIS STANDARD */

{
    int fdi;
    int nbytew, imsize; 
    int mode = 0644;
    int lastbs, n, nbpad;
    char *zpad;

    if ( bitpix != 16 && bitpix != 32 && bitpix != -32) {
        fprintf(stderr, "writeimg: bitpix = %5d not implemented\n", bitpix ); 
        exit(1);
    }
    fdi = open (fname, O_RDWR | O_CREAT, mode);
    if (fdi < 0) {
       fprintf(stderr, "writeimg: Error opening file %s\n", fname);
       exit(1);
    }
    if ( (nbytew = write (fdi, header, BSIZE)) != BSIZE) {
        fprintf(stderr, "writeimg: Error writing header in %s - returned %d bytes\n", fname, nbytew);
        exit(1);
    }
    imsize = abs(bitpix)*naxis1*naxis2/8;
    if (naxis > 2) imsize *= naxis3;
    if (naxis > 3) imsize *= naxis4;
    lastbs = imsize%BSIZE;             /* byte size of last block */
    if (bitpix == 16 && 12345 != htons(12345)) swab(imarr, imarr, imsize);
    if (abs(bitpix) == 32 && 123456789 != htonl(123456789)) iswab(imarr, imarr, imsize); 
    if ( (nbytew = write (fdi, imarr, imsize)) != imsize) {
        fprintf(stderr, "writeimg: Error writing data in %s - returned %d bytes\n", 
                fname, nbytew);
        exit(1);
    }
    if (lastbs > 0) {
        nbpad = BSIZE-lastbs;
        zpad=(char *)malloc(nbpad);
        for (n=0; n < nbpad; n++) zpad[n] = 0;
        if ( (nbytew = write (fdi, zpad, nbpad)) != nbpad) {
          fprintf(stderr, "writeimg: Error writing zero-pad in %s - returned %d bytes\n", 
                  fname, nbytew);
        }
        free(zpad);
    }

    close (fdi); /* Close disk file. */
}


void make_min_hdr(char *header)         /* creates minimal FITS header */
/* 11/15/95: Added bscale, b_zero keywords                   */
/*  2/11/99: Adds naxis3, naxis4 if naxis indicates it's necessary     */

{
    int nhb;
    int offset;

/* initialize  header array to asci blanks */
    for (nhb = 0; nhb < BSIZE; nhb++) 
        header[nhb] = ' ';

    offset = 0;
    sprintf(header,          "SIMPLE  =%21c", simple);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "BITPIX  =%21i", bitpix);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NAXIS   =%21i",  naxis);
    header[offset + 30] = ' ';

    offset += 80;
    sprintf(header + offset, "NAXIS1  =%21i", naxis1);
    header[offset + 30] = ' ';

    if (naxis > 1) {
        offset += 80;
        sprintf(header + offset, "NAXIS2  =%21i", naxis2);
        header[offset + 30] = ' ';
    }
    
    if (naxis > 2) {
        offset += 80;
        sprintf(header + offset, "NAXIS3  =%21i", naxis3);
        header[offset + 30] = ' ';
    }

    if (naxis > 3) {
        offset += 80;
        sprintf(header + offset, "NAXIS4  =%21i", naxis4);
        header[offset + 30] = ' ';
    }

    if (bscale != 0.) {
        offset += 80;
        sprintf(header + offset, "BSCALE  =%21.10e", bscale);
        header[offset + 30] = ' ';

        offset += 80;
        sprintf(header + offset, "BZERO   =%21.10e", b_zero);
        header[offset + 30] = ' ';
    }

    if (daxis1 > 0.) {
        offset += 80;
        sprintf(header + offset, "DAXIS1  =%21.10e", daxis1);
        header[offset + 30] = ' ';
    }

    if (daxis2 > 0. && naxis > 1) {
        offset += 80;
        sprintf(header + offset, "DAXIS2  =%21.10e", daxis2);
        header[offset + 30] = ' ';
    }

    if (daxis3 > 0. && naxis > 2) {
        offset += 80;
        sprintf(header + offset, "DAXIS3  =%21.10e", daxis3);
        header[offset + 30] = ' ';
    }

    if (daxis4 > 0. && naxis > 3) {
        offset += 80;
        sprintf(header + offset, "DAXIS4  =%21.10e", daxis4);
        header[offset + 30] = ' ';
    }

    offset += 80;
    sprintf(header + offset, "END");
    header[offset + 3] = ' ';

}
void imstats(short *pimage, double *min, double *max, double *mean, double *stdev)
/* Pixel statistics of short FITS image */
{
    int n, isize;
    float value=0., sum, sum2, num;
    int imin, imax;
    int scale;
    double sqrt();
    if (bscale == 0.) {
        scale = 0;
        bscale = 1.;
        b_zero  = 0.;
    }
    else {
        scale=1;
    }
    sum = 0.;
    sum2 = 0.;
    num = 0.;
    imin = pimage[0] ;
    imax = pimage[0] ;
    isize = naxis1*naxis2;
    for (n= 0; n < isize; n++) { 
        if ( pimage[n] < imin) {
            imin = pimage[n] ;
        }
        if ( pimage[n] > imax) {
            imax = pimage[n] ;
        }
        if (scale==1) 
             value = b_zero+(bscale*(double)pimage[n]);
        if (scale==0) 
             value = (double)pimage[n];
        sum  += value;
        sum2 += value*value;
        num += 1.;
    }
    *max = b_zero+(bscale*(double)(imax));
    *min = b_zero+(bscale*(double)(imin));
    *mean = sum/num;
    if (num > 1.) { 
        *stdev = sqrt( (sum2 - num*(*mean)*(*mean))/ (num - 1.) );
    } 
    else {
        *stdev = 0.;
    }
}
void imstati(int *pimage, double *min, double *max, double *mean, double *stdev)
/* Pixel statistics of int FITS image */
{
    int n, isize;
    double value=0., sum, sum2, num;
    int imin, imax;
    int scale;
    double sqrt();
    if (bscale == 0.) {
        scale = 0;
        bscale = 1.;
        b_zero  = 0.;
    }
    else {
        scale=1;
    }
    sum = 0.;
    sum2 = 0.;
    num = 0.;
    imin = pimage[0] ;
    imax = pimage[0] ;
    isize=naxis1*naxis2;
    for (n= 0; n < isize; n++) {
        if ( pimage[n] < imin) {
                imin = pimage[n] ;
        }
        if ( pimage[n] > imax) {
            imax = pimage[n] ;
        }
        if (scale==1) 
             value = b_zero+(bscale*(double)pimage[n]);
        if (scale==0) 
             value = (double)pimage[n];
        sum  += value;
        sum2 += value*value;
        num += 1.;
    }
    *max = b_zero+(bscale*(double)(imax));
    *min = b_zero+(bscale*(double)(imin));
    *mean = sum/num;
    if (num > 1.) {
        *stdev = sqrt( (sum2 - num*(*mean)*(*mean))/ (num - 1.) );
    }
    else {
        *stdev = 0.;
    }
}

void imstatf(float *pimage, double *min, double *max, double *mean, double *stdev)
/* Pixel statistics of float FITS image */
{
    int n, isize;
    double value, sum, sum2, num;
    float imin, imax;
    double sqrt();
    sum = 0.;
    sum2 = 0.;
    num = 0.;
    imin = pimage[0] ;
    imax = pimage[0] ;
    isize=naxis1*naxis2;
    for (n= 0; n < isize; n++) {
        if ( pimage[n] < imin) {
            imin = pimage[n] ;
        }
        if ( pimage[n] > imax) {
            imax = pimage[n] ;
        }
        value = (double)pimage[n];
        sum  += value;
        sum2 += value*value;
        num += 1.;
    }
    *max = (double)(imax);
    *min = (double)(imin);
    *mean = sum/num;
    if (num > 1.) {
        *stdev = sqrt( (sum2 - num*(*mean)*(*mean))/ (num - 1.) );
    }
    else {
        *stdev = 0.;
    }
}

/* 4 byte reversal for reading & writing 32 bit FITS images on little endian*/

void iswab(const void *f, void *t, int nbytes)
{
    const char *from= (const char *)f;
    char *to= (char *)t;
    int i;
    char tmp0, tmp1, tmp2, tmp3;
    for(i=0; i< nbytes; i+=4) {
        tmp0 = from[i];
        tmp1 = from[i+1];
        tmp2 = from[i+2];
        tmp3 = from[i+3];
        to[i]   = tmp3;
        to[i+1] = tmp2;
        to[i+2] = tmp1;
        to[i+3] = tmp0;
    }
}

/* rescales short image to specified minimum and maximum value */
/* sets zero or uniform images to maximum value (blank frame) */

void rescales(short *pimi, short *pimo, short omin, short omax)
{
    int n, npix, scale=0, blank;
    float value=0., ovalue, slope;
    float imin, imax;
    imin=9.e10;
    imax=-9.e10;

    if (bscale != 0.) scale=1;
    npix= naxis1*naxis2;
    for (n=0; n < npix; n++) {
        if (scale) value = b_zero+(bscale*(float)pimi[n]);
        else value=(float)pimi[n];
        if (value > imax) imax=value;
        if (value < imin) imin=value;
    }
    if (imax > imin) {
        slope = (float)(omax-omin)/(imax-imin);
        blank =0;
    }
    else  blank=1;
    for (n=0; n < npix; n++) {
        if (blank) pimo[n] = omax;
        else {
            if (scale) value = b_zero+(bscale*(float)pimi[n]);
            else value=(float)pimi[n];
            ovalue = (float)(omin) + slope*(value - imin);
            pimo[n] = rint(ovalue);
        }
    }
}


/* rescales int image to specified minimum and maximum value */
/* sets zero or uniform images to maximum value (blank frame) */

void rescalei(int *pimi, int *pimo, int omin, int omax)
{
    int n, npix, scale=0, blank;
    float value, ovalue, slope;
    float imin, imax;
    imin=9.e10;
    imax=-9.e10;

    if (bscale != 0.) scale=1;
    npix= naxis1*naxis2;
    for (n=0; n < npix; n++) {
        if (scale) value = b_zero+(bscale*(float)pimi[n]);
        else value=(float)pimi[n];
        if (value > imax) imax=value;
        if (value < imin) imin=value;
    }
    if (imax > imin) {
        slope = (float)(omax-omin)/(imax-imin);
        blank =0;
    }
    else  blank=1;
    for (n=0; n < npix; n++) {
        if (blank) pimo[n] = omax; 
        else {
            if (scale) value = b_zero+(bscale*(float)pimi[n]);
            else value=(float)pimi[n];
            ovalue = (float)(omin) + slope*(value - imin); 
            pimo[n] = rint(ovalue);
        }
    }
}

/* rescales short image to specified minimum and maximum value */
/* truncates input image beyond range given by imin - imax  */

void rescale2s(short *pimi, short *pimo, float imin, float imax, short omin, short omax)
{
    int n, npix, scale=0;
    float value, ovalue, slope;

    if (bscale != 0.) scale=1;
    npix= naxis1*naxis2;
    if (imin==imax) {
        fprintf(stderr, "rescale2s: imax = imin = %10.5f\n", imax);
        exit(1);
    }
    slope = (float)(omax-omin)/(imax-imin);
    for (n=0; n < npix; n++) {
        if (scale) value = b_zero+(bscale*(float)pimi[n]);
        else value=(float)pimi[n];
        if (value < imin) value = imin;
        if (value > imax) value = imax;
        ovalue = (float)(omin) + slope*(value - imin);
        pimo[n] = rint(ovalue);
    }
}


/* rescales int image to specified minimum and maximum value */
/* truncates input image beyond range given by imin - imax  */

void rescale2i(int *pimi, int *pimo, float imin, float imax, int omin, int omax)
{
    int n, npix, scale=0;
    float value, ovalue, slope;

    if (bscale != 0.) scale=1;
    npix= naxis1*naxis2;
    if (imin==imax) {
        fprintf(stderr, "rescale2l: imax = imin = %10.5f\n", imax);
        exit(1);
    }
    slope = (float)(omax-omin)/(imax-imin);
    for (n=0; n < npix; n++) {
        if (scale) value = b_zero+(bscale*(float)pimi[n]);
        else value=(float)pimi[n];
        if (value < imin) value = imin;
        if (value > imax) value = imax;
        ovalue = (float)(omin) + slope*(value - imin); 
        pimo[n] = rint(ovalue);
    }
}

/* updates keyword in header array                          */
/*  use sprintf to write value into char string "new_value" */
/*   e.g. sprintf(new_value, "%20.10f", pi);                */
/*                                                          */
int update_kwd(char *header, char *kwd, char *new_value)
{
    int nb, offset;
    nb=findkwd(kwd, header);
    if (nb > 1) {
        offset=(nb-1)*80;
        sprintf(header + offset, "%-8s", kwd);
        sprintf(header + offset+8, "=%21s", new_value);
        header[offset + 30] = ' ';
        return(1);
    }
    else {
/*        fprintf(stderr, "No %s keyword found; appending before END...\n", kwd); */
        nb=endline(header);
        if (nb < 0 || nb==36) {
            fprintf(stderr, "No END keyword found or header is full- not appending to this header! nb = %d\n", nb);
            return(-1);
        }
        offset=(nb-1)*80;
        sprintf(header + offset, "%-8s", kwd);
        sprintf(header + offset+8, "=%21s", new_value);
        header[offset + 30] = ' ';
        offset += 80;
        sprintf(header + offset, "END");
        header[offset + 3] = ' ';
        return(1);
    }
}

/* returns optimal bscale, b_zero for scaling float array to integer */

void set_scale(float *pfimage, float *bscale_out, float *b_zero_out)
{
    float vmin, vmax, vrange, bs, bst;
    int i, iexp, numpix;
    float epsilon=1.e-6;
    vmin=9.e99;
    vmax=-9.e99;
    numpix = naxis1*naxis2;
    if (naxis>2) numpix *=naxis3;
    if (naxis>3) numpix *=naxis4;
    if (naxis>4) {
        fprintf(stderr, "set_scale: naxis = %10d not implemented\n", naxis);
        exit(1);
    }
    for (i=0; i < numpix; i++) {
        if (pfimage[i] < vmin) vmin=pfimage[i];
        if (pfimage[i] > vmax) vmax=pfimage[i];
    }
    if (vmin <= 0. && vmax >= 0.){
/* vmin, vmax straddle zero, set vrange to 0 to maximum abs(value) */
        vrange=fabs(vmax);
        if (fabs(vmin) > vrange) vrange=fabs(vmin);
        *b_zero_out=0.;
    }
    else {
/* set b_zero_out to vmin, set vrange to max - min */
        vrange=(vmax-vmin);
        *b_zero_out=vmin;
    }

    if ((vrange/fabs(vmax)) > epsilon) {
        iexp=(int)(log10(vrange));
        bs=pow(10.,(float)(iexp)-3.);
        for (i=2; i< 100; i+=2) {
            bst = bs*(1. - (float)(i)/100.);
            if ((vrange/bst) < 32768) *bscale_out = bst;
        }
    }
    else {
        *bscale_out = 1.;
        *b_zero_out = 0.;
    }
}

/* returns optimal bscale, b_zero for scaling double array to integer */

void set_scaled(double *pdimage, float *bscale_out, float *b_zero_out)
{
    float vmin, vmax, vrange, bs, bst;
    int i, iexp, numpix;
    float epsilon=1.e-6;
    vmin=9.e99;
    vmax=-9.e99;
    numpix = naxis1*naxis2;
    if (naxis>2) numpix *=naxis3;
    if (naxis>3) numpix *=naxis4;
    if (naxis>4) {
        fprintf(stderr, "set_scaled: naxis = %10d not implemented\n", naxis);
        exit(1);
    }
    for (i=0; i < numpix; i++) {
        if (pdimage[i] < vmin) vmin=pdimage[i];
        if (pdimage[i] > vmax) vmax=pdimage[i];
    }
printf("vmin = %20.5e vmax = %20.5e\n", vmin, vmax);
    if (vmin <= 0. && vmax >= 0.){
/* vmin, vmax straddle zero, set vrange to 0 to maximum abs(value) */
        vrange=fabs(vmax);
        if (fabs(vmin) > vrange) vrange=fabs(vmin);
        *b_zero_out=0.;
    }
    else {
/* set b_zero_out to vmin, set vrange to max - min */
        vrange=(vmax-vmin);
        *b_zero_out=vmin;
    }

    if ((vrange/fabs(vmax)) > epsilon) {
        iexp=(int)(log10(vrange));
        bs=pow(10.,(float)(iexp)-3.);
        for (i=2; i< 100; i+=2) {
            bst = bs*(1. - (float)(i)/100.);
            if ((vrange/bst) < 32768) *bscale_out = bst;
        }
    }
    else {
        *bscale_out = 1.;
        *b_zero_out = 0.;
    }
}
